<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Dtos\Post;

use Erlage\Photogram\Data\Dtos\AbstractDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintValidatorDTO;
use Erlage\Photogram\Data\Dtos\Traits\TraitComplaintTransformerDTO;

final class PostDisplayContentItemDTO extends AbstractDTO
{
    /**
     * to comply with DTO interface
     */
    use TraitComplaintValidatorDTO;
    use TraitComplaintTransformerDTO;

    /*
    |--------------------------------------------------------------------------
    | private key map, warning! must not change
    |--------------------------------------------------------------------------
    */

    const KEY_DISPLAY_ITEM = 'display_item';

    const KEY_DISPLAY_USER_TAGS = 'user_tags';

    /*
    |--------------------------------------------------------------------------
    | data
    |--------------------------------------------------------------------------
    */

    /**
     * @var PostDisplayItemDTO
     */
    private $displayItemDTO;

    /**
     * @var PostDisplayUserTagsOnItemDTO
     */
    private $displayUserTagsOnItemDTO;

    /*
    |--------------------------------------------------------------------------
    | getters
    |--------------------------------------------------------------------------
    */

    public function getDisplayItemDTO(): PostDisplayItemDTO
    {
        return $this -> displayItemDTO;
    }

    public function getDisplayUserTagsOnItemDTO(): PostDisplayUserTagsOnItemDTO
    {
        return $this -> displayUserTagsOnItemDTO;
    }

    public function getNumberOfUsersTagged(): int
    {
        return $this -> displayUserTagsOnItemDTO -> getLength();
    }

    /*
    |--------------------------------------------------------------------------
    | setters
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public function setDisplayItemDTO(PostDisplayItemDTO $displayItemDTO)
    {
        $this -> displayItemDTO = $displayItemDTO;

        return $this;
    }

    /**
     * @return static 
     */
    public function setDisplayUserTagsOnItemDTO(
        PostDisplayUserTagsOnItemDTO $displayUserTagsOnItemDTO
    ) {
        $this -> displayUserTagsOnItemDTO = $displayUserTagsOnItemDTO;

        return $this;
    }

    /*
    |--------------------------------------------------------------------------
    | from json map
    |--------------------------------------------------------------------------
    */

    /**
     * @return static 
     */
    public static function fromJsonMap(array $data)
    {
        return (new static())
            -> setDisplayItemDTO(
                PostDisplayItemDTO::fromJsonMap($data[self::KEY_DISPLAY_ITEM])
            )
            -> setDisplayUserTagsOnItemDTO(
                PostDisplayUserTagsOnItemDTO::fromJsonMap($data[self::KEY_DISPLAY_USER_TAGS])
            );
    }

    /*
    |--------------------------------------------------------------------------
    | serialization for json_encode
    |--------------------------------------------------------------------------
    */

    public function jsonSerialize()
    {
        return array(
            self::KEY_DISPLAY_ITEM      => $this -> displayItemDTO,
            self::KEY_DISPLAY_USER_TAGS => $this -> displayUserTagsOnItemDTO,
        );
    }

    /*
    |--------------------------------------------------------------------------
    | implement DTO methods
    |--------------------------------------------------------------------------
    */

    /**
     * @param PostDisplayContentItemDTO $displayContentItem
     */
    public static function serialize($displayContentItem): string
    {
        return \json_encode($displayContentItem);
    }

    /**
     * @return PostDisplayContentItemDTO
     */
    public static function deSerialize(string $displayContentItem)
    {
        $data = \json_decode($displayContentItem, true);

        return self::fromJsonMap($data);
    }
}
